let data = [];

// جلب البيانات مع إضافة تايت مستامب لمنع الكاش عند البداية
fetch("data.json?v=" + new Date().getTime()).then(res => res.json()).then(json => data = json);

function containsArabic(text) {
  const arabicRegex = /[\u0600-\u06FF]/;
  return arabicRegex.test(text);
}

function getInitialSuggestions() {
  let shuffled = [...data].sort(() => 0.5 - Math.random());
  let picked = [],
    contexts = new Set();
  for (let item of shuffled) {
    if (!contexts.has(item.context)) {
      picked.push(item.keyword);
      contexts.add(item.context);
    }
    if (picked.length >= 5) break;
  }
  return picked;
}

const chatIcon = document.getElementById("chat-icon");
const chatWindow = document.getElementById("chat-window");
const chatBox = document.getElementById("chat-box");
const input = document.getElementById("user-input");
const chatHeader = document.getElementById("chat-header");

chatIcon.onclick = toggleChat;

function toggleChat() {
  chatWindow.style.display = chatWindow.style.display === "flex" ? "none" : "flex";
  if (chatWindow.style.display === "flex") {
    const greeting = containsArabic(input.value) ? "مرحباً! إليك بعض المقترحات للبدء:" : "Hi! Here are some suggestions to start:";
    showBotMessage(greeting, getInitialSuggestions());
  }
}

document.addEventListener("click", e => {
  if (!chatWindow.contains(e.target) && !chatIcon.contains(e.target)) chatWindow.style.display = "none";
});

chatHeader.addEventListener("click", e => {
  if (!e.target.closest("button")) chatWindow.style.display = "none";
});

input.addEventListener("keypress", e => {
  if (e.key === "Enter") sendMessage();
});

function levenshtein(a, b) {
  const matrix = Array.from({
    length: b.length + 1
  }, (_, i) => [i]);
  for (let i = 0; i <= a.length; i++) matrix[0][i] = i;
  for (let i = 1; i <= b.length; i++) {
    for (let j = 1; j <= a.length; j++) {
      if (b[i - 1] === a[j - 1]) matrix[i][j] = matrix[i - 1][j - 1];
      else matrix[i][j] = Math.min(matrix[i - 1][j - 1] + 1, matrix[i][j - 1] + 1, matrix[i - 1][j] + 1);
    }
  }
  return matrix[b.length][a.length];
}

function findBestMatch(inputText) {
  inputText = inputText.toLowerCase();
  let bestScore = Infinity,
    bestItem = null;
  const isArabicInput = containsArabic(inputText);

  for (let item of data) {
    const isArabicKeyword = containsArabic(item.keyword);
    if (isArabicInput !== isArabicKeyword) continue;

    const texts = [item.keyword, ...(item.aliases || [])].map(t => t.toLowerCase());
    for (let t of texts) {
      if (t.includes(inputText)) return item;
      const score = levenshtein(inputText, t);
      if (score < bestScore) {
        bestScore = score;
        bestItem = item;
      }
    }
  }
  if (bestScore <= 2) return { ...bestItem, bestScore };
  return null;
}

function sendMessage(text = null) {
  let userText = text || input.value.trim();
  if (!userText) return;
  addMessage(userText, "user");
  input.value = "";

  if (userText.length < 3) {
    botReply(containsArabic(userText) ? "نعتذر، لم أفهم ما تقصده." : "❌ Sorry, I didn't understand what you mean.", [], null, true);
    saveUnknown(userText);
    return;
  }

  let found = findBestMatch(userText);
  if (found) {
    const responseText = Array.isArray(found.response) ? found.response[Math.floor(Math.random() * found.response.length)] : found.response;
    botReply(responseText, found.suggestions, found.source_url);
  } else {
    // رسالة انتظار مع تحويل السؤال
    const waitMsg = containsArabic(userText) ? "سؤالك غير مسجل لدينا، جاري تحويله للمختص للرد عليك حالاً..." : "Question not found, forwarding to a specialist for a live reply...";
    botReply(waitMsg, [], null, false);
    saveUnknown(userText, null, containsArabic(userText) ? "ar" : "en");
    
    // بدء البحث عن ردك من تلجرام
    startWaitingForAdmin(userText);
  }
}

function addMessage(text, type) {
  const msg = document.createElement("div");
  msg.classList.add(type + "-message");
  msg.textContent = text;
  chatBox.appendChild(msg);
  chatBox.scrollTop = chatBox.scrollHeight;
}

function botReply(message, suggestions = [], source_url = null, isError = false) {
  if (typeof message !== "string") message = String(message);
  const typing = document.createElement("div");
  typing.classList.add("bot-message");
  typing.innerHTML = '<div class="typing-indicator"><span></span><span></span><span></span></div>';
  chatBox.appendChild(typing);
  chatBox.scrollTop = chatBox.scrollHeight;

  setTimeout(() => {
    typing.remove();
    const msg = document.createElement("div");
    msg.classList.add("bot-message");
    chatBox.appendChild(msg);
    let i = 0;
    let interval = setInterval(() => {
      msg.textContent += message.charAt(i);
      i++;
      if (i >= message.length) {
        clearInterval(interval);
        if (suggestions && suggestions.length > 0) {
          const suggBox = document.createElement("div");
          suggBox.classList.add("suggestions");
          suggestions.forEach(s => {
            const btn = document.createElement("button");
            btn.textContent = s;
            btn.onclick = () => sendMessage(s);
            suggBox.appendChild(btn);
          });
          chatBox.appendChild(suggBox);
        }
        if (source_url) {
          const link = document.createElement("div");
          link.classList.add("source-link");
          link.textContent = "Go to Source";
          link.onclick = () => window.open(source_url, "_blank");
          chatBox.appendChild(link);
        }
        if (isError) {
          const supportBtn = document.createElement("a");
          supportBtn.classList.add("contact-support");
          supportBtn.textContent = "Contact Support";
          supportBtn.href = "mailto:support@example.com";
          supportBtn.target = "_blank";
          chatBox.appendChild(supportBtn);
        }
      }
      chatBox.scrollTop = chatBox.scrollHeight;
    }, 15);
  }, 500);
}

// الوظيفة الجديدة: فحص ملف JSON بانتظار رد المسؤول
function startWaitingForAdmin(userQuestion) {
    let checkCount = 0;
    const interval = setInterval(() => {
        // فحص الملف الأساسي والملف المؤقت
        Promise.all([
            fetch('data.json?v=' + Date.now()).then(res => res.json()),
            fetch('temp_chat.json?v=' + Date.now()).catch(() => []).then(res => res ? res.json() : [])
        ]).then(([permanentData, temporaryData]) => {
            // دمج البيانات للفحص فقط دون التأثير على ملف data.json الأصلي
            const combinedData = [...permanentData, ...temporaryData];
            const match = combinedData.find(item => item.keyword.toLowerCase() === userQuestion.toLowerCase());
            
            if (match) {
                botReply(Array.isArray(match.response) ? match.response[0] : match.response);
                clearInterval(interval);
            }
        });

        checkCount++;
        if (checkCount > 60) clearInterval(interval);
    }, 3000);
}

function saveUnknown(message, bestScore = null, language = null) {
  fetch('save_unknown.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: 'message=' + encodeURIComponent(message) + '&score=' + encodeURIComponent(bestScore) + '&lang=' + encodeURIComponent(language)
  });
}

function showBotMessage(message, suggestions = []) { botReply(message, suggestions); }
function resetChat() { chatBox.innerHTML = ""; }