
// Canvas background
const canvas = document.getElementById('interactive-bg');
const ctx = canvas.getContext('2d');
let particles = [];
let mouse = { x: null, y: null, radius: 100 };

canvas.width = window.innerWidth;
canvas.height = window.innerHeight;

window.addEventListener('resize', () => {
    canvas.width = window.innerWidth;
    canvas.height = window.innerHeight;
    init();
});

window.addEventListener('mousemove', (e) => {
    mouse.x = e.x;
    mouse.y = e.y;
});

// Particle constructor
function Particle(x, y, directionX, directionY, size, color) {
    this.x = x;
    this.y = y;
    this.directionX = directionX;
    this.directionY = directionY;
    this.size = size;
    this.color = color;
}

// Draw particle
Particle.prototype.draw = function() {
    ctx.beginPath();
    ctx.arc(this.x, this.y, this.sizeGlow || this.size, 0, Math.PI * 2, false);
    ctx.fillStyle = `rgba(255, 255, 255, ${this.brightness || 1})`;
    ctx.fill();
};

// Update particle
Particle.prototype.update = function() {
    // Bounce on screen edges
    if (this.x + this.size > canvas.width || this.x - this.size < 0) this.directionX = -this.directionX;
    if (this.y + this.size > canvas.height || this.y - this.size < 0) this.directionY = -this.directionY;

    this.x += this.directionX;
    this.y += this.directionY;

    // Mouse repulsion
    if(mouse.x && mouse.y){
        let dxMouse = mouse.x - this.x;
        let dyMouse = mouse.y - this.y;
        let distMouse = Math.sqrt(dxMouse*dxMouse + dyMouse*dyMouse);
        if(distMouse < mouse.radius + this.size){
            this.x -= (dxMouse / distMouse) * 2;
            this.y -= (dyMouse / distMouse) * 2;
        }
    }

    // Hero image repulsion (oval)
    let heroImg = document.querySelector('.hero-image img');
    if(heroImg){
        let rect = heroImg.getBoundingClientRect();
        let imgX = rect.left + rect.width/2 + window.scrollX;
        let imgY = rect.top + rect.height/6 + window.scrollY;
        let rx = rect.width/2 + 20;
        let ry = rect.height/2 + 20;

        let dx = this.x - imgX;
        let dy = this.y - imgY;

        if ((dx*dx)/(rx*rx) + (dy*dy)/(ry*ry) < 1){
            this.x += (dx/rx) * 3;
            this.y += (dy/ry) * 3;
        }
    }

    // Light effect behind image
    let lightX = heroImg ? heroImg.getBoundingClientRect().left + heroImg.width/2 + window.scrollX : canvas.width/2;
    let lightY = heroImg ? heroImg.getBoundingClientRect().top + heroImg.height/2 + window.scrollY : canvas.height/2;
    let dxLight = this.x - lightX;
    let dyLight = this.y - lightY;
    let distLight = Math.sqrt(dxLight*dxLight + dyLight*dyLight);

    let maxDist = 300; // تأثير الضوء
    let factor = 1 - Math.min(distLight / maxDist, 1); // 0..1
    this.brightness = 0.2 + factor * 0.8; // شعيرات أقرب = أكثر لمعاناً
    this.sizeGlow = this.size + factor * 2; // زيادة حجم الشعيرة قرب الضوء
};


// Init particles
function init() {
    particles = [];
    for (let i = 0; i < 200; i++) {
        let size = (Math.random() * 5) + 1;
        let x = Math.random() * (canvas.width - size*2) + size;
        let y = Math.random() * (canvas.height - size*2) + size;
        let directionX = (Math.random() * 0.5) - 0.25;
        let directionY = (Math.random() * 0.5) - 0.25;
        let color = '#fff';
        particles.push(new Particle(x, y, directionX, directionY, size, color));
    }
}

// Animate particles
function animate() {
    requestAnimationFrame(animate);
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    for (let i = 0; i < particles.length; i++) {
        particles[i].update();
        particles[i].draw();
    }
    connect();
}

// Connect particles with lines
function connect() {
    let opacityValue = 1;
    for (let a = 0; a < particles.length; a++) {
        for (let b = a; b < particles.length; b++) {
            let distance = ((particles[a].x - particles[b].x)**2) + ((particles[a].y - particles[b].y)**2);
            if (distance < (canvas.width/7)*(canvas.height/7)) {
                opacityValue = 1 - (distance / 20000);
                ctx.strokeStyle = `rgba(255, 255, 255, ${opacityValue})`;
                ctx.lineWidth = 1;
                ctx.beginPath();
                ctx.moveTo(particles[a].x, particles[a].y);
                ctx.lineTo(particles[b].x, particles[b].y);
                ctx.stroke();
            }
        }
    }
}

init();
animate();
